<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Mail\Ordercreated;
use App\Models\BranchModel;
use App\Models\ClientAddressModel;
use App\Models\ClientModel;
use App\Models\OrdersItemsModel;
use App\Models\OrdersModel;
use App\Models\ProductModel;
use App\Models\RightsModel;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use Mpdf\Mpdf;

class MobOrderController extends Controller
{
    public function index(Request $request)
    {
        $user = Auth::user();

        $query = OrdersModel::with([
            'client' => function ($query) {
                $query->select('id', 'user_id');
            },
            'client.user' => function ($query) {
                $query->select('id', 'name', 'phone');
            },
            'branch',
            'ordersItem'
        ])->orderBy('id', 'desc');

        if ($request->filled('branch_id')) {
            $query->where('branch_id', $request->branch_id);
        }

        if ($user->user_flg === 'C') {
            $client_id = ClientModel::where('user_id', $user->id) // Corrected: use $user->id, not $user->user_id
                ->when($request->filled('branch_id'), function ($q) use ($request) {
                    $q->where('branch_id', $request->branch_id);
                })
                ->pluck('id')
                ->first();

            if ($client_id) {
                $query->where('client_id', $client_id);
            } else {
                return response()->json([
                    'status' => false,
                    'message' => 'Invalid client/user',
                ], 400);
            }
        }

        if ($request->filled('search')) {
            $searchTerm = $request->search;

            $query->where(function ($q) use ($searchTerm) {
                $q->where('order_id', 'like', "%{$searchTerm}%")
                    ->orWhereHas('client.user', function ($userQuery) use ($searchTerm) {
                        $userQuery->where('name', 'like', "%{$searchTerm}%")
                            ->orWhere('phone', 'like', "%{$searchTerm}%");
                    });
            });
        }

        $perPage = $request->get('per_page', 10);

        return response()->json($query->paginate($perPage));
    }


    public function getClientAddress($phone)
    {
        $user_id = User::where('phone', $phone)->value('id');
        $client = ClientModel::with('clientAddress')->where('user_id', $user_id)->value('id');
        $clientID = isset($client) ? $client : $phone;
        $client_address = ClientAddressModel::where('client_id', $clientID)->get();

        return response()->json($client_address);
    }

    public function getClients(Request $request)
    {
        if (Auth::user()->user_flg == 'C') {
            $clients = ClientModel::with(['user:id,name,phone'])
                ->where('user_id', Auth::user()->id)
                ->get();
        } else {
            $clients = ClientModel::with(['user:id,name,phone'])
                ->orderBy('id', 'desc')
                ->get();
        }
        return response()->json($clients);
    }

    public function store(Request $request)
    {
        $authUser = Auth::user();
        $request->validate([
            'products.*.qty' => 'required|numeric',
            'products.*.unit_price' => 'required|numeric',
            'products.*.amount' => 'required|numeric',
            'client_num' => 'required',
        ]);

        DB::beginTransaction();

        try {
            $user = User::where('phone', $request->client_num)->first();

            $client_id = $this->saveClient($request, $user);

            if (!$client_id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to create client'
                ], 400);
            }

            $address_id = null;
            if ($request->has('address_id')) {
                $address_id = $request->address_id;
            } elseif ($request->has('address')) {
                $address = ClientAddressModel::create([
                    'client_id' => $client_id,
                    'address' => $request->address,
                    'is_default' => $request->save_address ? 'Y' : 'N',
                    'oprtnl_flag' => 'A',
                    'created_by' => $authUser->id
                ]);
                $address_id = $address->id;
            }

            $lastOrder = OrdersModel::orderBy('id', 'desc')->first();
            $order_id = $this->generateOrderId($lastOrder);

            $order = OrdersModel::create([
                'order_id' => $order_id,
                'client_id' => $client_id,
                'order_date' => $request->order_date,
                'due_date' => $request->due_date,
                'amount' => $request->sub_total,
                'final_amount' => $request->total_amount,
                'discount_type' => $request->discount_type ?? null,
                'discount' => $request->discount ?? null,
                'note' => $request->note,
                'term' => $request->term,
                'branch_id' => $request->branch_id,
                'status' => $request->status ?? '',
                'address_id' => $address_id
            ]);

            foreach ($request->products as $field) {
                $product = ProductModel::find($field['id']);

                OrdersItemsModel::create([
                    'order_id' => $order->id,
                    'product_id' => $field['id'],
                    'product_name' => $product->name ?? '',
                    'available_quantity' => $product->qty,
                    'quantity' => $field['qty'],
                    'price' => $field['unit_price'],
                    'total' => $field['amount'],
                ]);
            }
            $settings = BranchModel::where('id', $request->branch_id)->first();

            $orderDetails = OrdersModel::with('client.clientAddress', 'ordersItem.product')
                ->where('id', $order->id)->where('branch_id', $request->branch_id)->first();
            $data = [
                'addedOrder' => $orderDetails,
                'settings' => $settings
            ];

            $html = view('orders.email-pdf.order-generate-pdf_email', $data)->render();
            $mpdf = new Mpdf([
                'mode' => 'utf-8',
                'format' => 'A4',
                'default_font' => 'freeserif',
                'autoScriptToLang' => true,
                'autoLangToFont' => true,
            ]);

            $mpdf->WriteHTML($html);
            $pdfContent = $mpdf->Output('', 'S');
            $admin_mail = User::where('user_flg', 'A')->where('branch_id', $request->branch_id)->pluck('email')->toArray();
            Mail::to($admin_mail)->send(new Ordercreated($pdfContent, $order, $settings));

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Order created successfully',
                'data' => [
                    'order_id' => $order->id,
                    'order_number' => $order->order_id
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            \Log::error('Error creating order: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to create order',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function update(Request $request, $id)
    {
        $authUser = Auth::user();
        $request->validate([
            'products.*.qty' => 'required|numeric',
            'products.*.unit_price' => 'required|numeric',
            'products.*.amount' => 'required|numeric',
            'client_num' => 'required',
        ]);

        DB::beginTransaction();

        try {
            $order = OrdersModel::findOrFail($id);
            $user = User::where('phone', $request->client_num)->first();

            $client_id = $this->saveClient($request, $user);

            if (!$client_id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to update client information'
                ], 400);
            }

            $address_id = $order->address_id;
            if ($request->has('address_id')) {
                $address_id = $request->address_id;
            } elseif ($request->has('address')) {
                // If new address is provided
                $address = ClientAddressModel::create([
                    'client_id' => $client_id,
                    'address' => $request->address,
                    'is_default' => $request->save_address ? 'Y' : 'N',
                    'oprtnl_flag' => 'A',
                    'created_by' => $authUser->id
                ]);
                $address_id = $address->id;
            }

            $order->update([
                'client_id' => $client_id,
                'order_date' => $request->order_date,
                'due_date' => $request->due_date,
                'amount' => $request->sub_total,
                'final_amount' => $request->total_amount,
                'discount_type' => $request->discount_type ?? null,
                'discount' => $request->discount ?? null,
                'note' => $request->note,
                'term' => $request->term,
                'branch_id' => $request->branch_id,
                'status' => $request->status ?? '',
                'address_id' => $address_id
            ]);

            // Delete existing order items
            OrdersItemsModel::where('order_id', $order->id)->delete();

            // Add new order items
            foreach ($request->products as $field) {
                $product = ProductModel::find($field['id']);

                OrdersItemsModel::create([
                    'order_id' => $order->id,
                    'product_id' => $field['id'],
                    'product_name' => $product->name ?? '',
                    'available_quantity' => $product->qty,
                    'quantity' => $field['qty'],
                    'price' => $field['unit_price'],
                    'total' => $field['amount'],
                ]);
            }

            // Get updated order details
            $orderDetails = OrdersModel::with('client.clientAddress', 'ordersItem.product')
                ->where('id', $order->id)
                ->where('branch_id', $request->branch_id)
                ->first();

            $settings = BranchModel::where('id', $request->branch_id)->first();
            $data = [
                'addedOrder' => $orderDetails,
                'settings' => $settings
            ];

            // Generate updated PDF
            $html = view('orders.email-pdf.order-generate-pdf_email', $data)->render();
            $mpdf = new Mpdf([
                'mode' => 'utf-8',
                'format' => 'A4',
                'default_font' => 'freeserif',
                'autoScriptToLang' => true,
                'autoLangToFont' => true,
            ]);

            $mpdf->WriteHTML($html);
            $pdfContent = $mpdf->Output('', 'S');
            $admin_mail = User::where('user_flg', 'A')->where('branch_id', $request->branch_id)->pluck('email')->toArray();
            Mail::to($admin_mail)->send(new Ordercreated($pdfContent, $order, $settings));

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Order updated successfully',
                'data' => [
                    'order_id' => $order->id,
                    'order_number' => $order->order_id
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            \Log::error('Error updating order: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to update order',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    protected function generateOrderId($lastorder)
    {
        if ($lastorder) {
            $lastOrderId = $lastorder->order_id;
            $parts = explode('-', $lastOrderId);
            $prefix = $parts[0];
            $number = (int) $parts[1];
            $newNumber = str_pad($number + 1, strlen($parts[1]), '0', STR_PAD_LEFT);
            return $prefix . '-' . $newNumber;
        }
        return 'OSORD-001';
    }

    protected function saveClient(Request $request, $user = null)
    {
        try {
            if ($user) {
                $client = ClientModel::where('user_id', $user->id)->first();
                return $client->id;
            }

            $randomEmail = Str::random(10) . '@billing.com';

            $right = RightsModel::where('rights_type', 'client')->first();
            $clientRight = $right ? $right->id : null;
            $user = User::create([
                'name' => $request->client_id,
                'email' => $randomEmail,
                'phone' => $request->client_num,
                'password' => Hash::make('client@123'),
                'user_flg' => 'C',
                'active_status' => 'A',
                'branch_id' => session('branch_id'),
                'rights_id' => $clientRight,

            ]);

            $client = ClientModel::create([
                'user_id' => $user->id,
                'branch_id' => session('branch_id'),
                'active_status' => 'A',
            ]);
            $allBranchIds = DB::table('branch')->pluck('id');

            $insertData = [];

            foreach ($allBranchIds as $branchId) {
                $insertData[] = [
                    'user_id' => $user->id,
                    'branch_id' => $branchId,
                ];
            }

            DB::table('branch_user')->insert($insertData);
            return $client->id;
        } catch (\Exception $e) {
            \Log::info('While create Client:' . $e);
            return null;
        }
    }

    public function destroy($id)
    {
        DB::beginTransaction();

        try {
            $order = OrdersModel::findOrFail($id);

            // Delete all associated order items first
            OrdersItemsModel::where('order_id', $order->id)->delete();

            // Then delete the order
            $order->delete();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Order deleted successfully'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            \Log::error('Error deleting order: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to delete order',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
